/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.commons.numbers.gamma;

import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.CsvSource;

/**
 * Tests for {@link Gamma}.
 *
 * <p>The class directly calls the methods in {@link BoostGamma}. This test ensures
 * the arguments are passed through correctly. Accuracy of the function is tested
 * in {@link BoostGammaTest}.
 */
class GammaTest {
    /**
     * Reference data for the {@link Gamma#value(double)} function. This
     * data was generated with the following <a
     * href="http://maxima.sourceforge.net/">Maxima</a> script.
     * Note: This data is different from the data used for {@link BoostGammaTest}.
     *
     * <pre>
     * kill(all);
     *
     * fpprec : 64;
     *
     * EPSILON : 10**(-fpprec + 1);
     * isInteger(x) := abs(x - floor(x)) <= EPSILON * abs(x);
     * str(x) := ssubst("e","b",string(x));
     *
     * x : makelist(bfloat(i / 8) + 0.0625, i, -160, 160);
     * x : append(x, makelist(bfloat(i / 2) + 0.125, i, 41, 200));
     *
     * for i : 1 while i <= length(x) do if not(isInteger(x[i])) then
     *     printf(true, "\"~f,~a\",~%", x[i], str(gamma(x[i]))), fpprintprec:30;
     *</pre>
     */
    @ParameterizedTest
    @CsvSource(value = {
        "-19.9375,7.99347140318629900587250189872e-18",
        "-19.8125,4.09142642752295504387006593351e-18",
        "-19.6875,3.98176742212573192270008693047e-18",
        "-19.5625,4.91270643046446517527599456012e-18",
        "-19.4375,7.14422504112472659254903607729e-18",
        "-19.3125,1.22455052925828737010057593176e-17",
        "-19.1875,2.66094388924946694269102198385e-17",
        "-19.0625,1.09937522735384800858081253897e-16",
        "-18.9375,-1.59369836101026836429583006606e-16",
        "-18.8125,-8.10613860952985468066756813076e-17",
        "-18.6875,-7.83910461231003472281579614437e-17",
        "-18.5625,-9.61048195459610999913366435823e-17",
        "-18.4375,-1.38865874236861873142671888752e-16",
        "-18.3125,-2.36491320963006748350673726821e-16",
        "-18.1875,-5.10568608749741469628839843151e-16",
        "-18.0625,-2.09568402714327276635717390241e-15",
        "-17.9375,3.0180662711631957148852281876e-15",
        "-17.8125,1.5249673259178039118005862546e-15",
        "-17.6875,1.46493267442543773882620190448e-15",
        "-17.5625,1.7839457128219029185891864465e-15",
        "-17.4375,2.56033955624214078606801294887e-15",
        "-17.3125,4.33074731513506107917171262241e-15",
        "-17.1875,9.2859665716359229788745246473e-15",
        "-17.0625,3.78532927402753643423264536122e-14",
        "-16.9375,-5.4136563738989823135753780615e-14",
        "-16.8125,-2.71634804929108821789479426601e-14",
        "-16.6875,-2.59109966788999300054884461855e-14",
        "-16.5625,-3.13305465814346700077225869666e-14",
        "-16.4375,-4.4645921011972329957060975796e-14",
        "-16.3125,-7.49760628932757449331602747755e-14",
        "-16.1875,-1.59602550449992426199405892375e-13",
        "-16.0625,-6.45871807380948404090945114759e-13",
        "-15.9375,9.16938048329140129361829659167e-13",
        "-15.8125,4.56686015787064206633562285972e-13",
        "-15.6875,4.3238975707914258196658844572e-13",
        "-15.5625,5.18912177755011722002905346634e-13",
        "-15.4375,7.33867326634295173669189789646e-13",
        "-15.3125,1.22304702594656058922217698228e-12",
        "-15.1875,2.58356628540925239910288288283e-12",
        "-15.0625,1.03743159060564837407108059058e-11",
        "-14.9375,-1.4613700145245670811704160193e-11",
        "-14.8125,-7.22134762463295276739320364694e-12",
        "-14.6875,-6.78311431417904925460085624223e-12",
        "-14.5625,-8.07557076631236992367021445699e-12",
        "-14.4375,-1.13290768549169317435181173777e-11",
        "-14.3125,-1.87279075848067090224645850411e-11",
        "-14.1875,-3.9237912959653020811375033783e-11",
        "-14.0625,-1.56263133334975786344456513956e-10",
        "-13.9375,2.18292145919607207749830892882e-10",
        "-13.8125,1.0696621168987561286701182902e-10",
        "-13.6875,9.96269914895047859269500760578e-11",
        "-13.5625,1.1760049928442388701344749803e-10",
        "-13.4375,1.6356354709286320204704281964e-10",
        "-13.3125,2.68043177307546022884024373401e-10",
        "-13.1875,5.56687890115077232761383291796e-10",
        "-13.0625,2.19745031252309699546891972751e-9",
        "-12.9375,-3.04244678375452545801326806955e-9",
        "-12.8125,-1.47747079896640690272560088834e-9",
        "-12.6875,-1.36364444601259675737512916604e-9",
        "-12.5625,-1.59495677154499896761988169203e-9",
        "-12.4375,-2.19788516406034927750713788891e-9",
        "-12.3125,-3.5683247979067064296435744709e-9",
        "-12.1875,-7.34132155089258100704074216056e-9",
        "-12.0625,-2.87041947073329545033127639406e-8",
        "-11.9375,3.93616552648241731130466556498e-8",
        "-11.8125,1.89300946117570884411717613819e-8",
        "-11.6875,1.73012389087848213591969512941e-8",
        "-11.5625,2.00366444425340495307247637561e-8",
        "-11.4375,2.73361967280005941389950274934e-8",
        "-11.3125,4.39349990742263229149865106729e-8",
        "-11.1875,8.94723564015033310233090450818e-8",
        "-11.0625,3.46244348657203763696210215033e-7",
        "-10.9375,-4.69879759723838566536994451819e-7",
        "-10.8125,-2.23611742601380607211341431323e-7",
        "-10.6875,-2.0220822974642259963561436825e-7",
        "-10.5625,-2.3167370136679994769900508093e-7",
        "-10.4375,-3.12657750076506795464755626955e-7",
        "-10.3125,-4.97014677027185277975784901988e-7",
        "-10.1875,-1.00097198724181851582326994185e-6",
        "-10.0625,-3.83032810702031663588932550381e-6",
        "-9.9375,5.13930987197948432149837681677e-6",
        "-9.8125,2.41780196687742781547262922619e-6",
        "-9.6875,2.16110045541489153360562856067e-6",
        "-9.5625,2.44705347068682444757074116733e-6",
        "-9.4375,3.26336526642353967766338685634e-6",
        "-9.3125,5.12546385684284817912528180175e-6",
        "-9.1875,1.01974021200260261299495625326e-5",
        "-9.0625,3.85426765768919361486363378821e-5",
        "-8.9375,-5.10718918527961254448901196167e-5",
        "-8.8125,-2.37246817999847604393251742819e-5",
        "-8.6875,-2.09356606618317617318045266815e-5",
        "-8.5625,-2.33999488134427587798952124126e-5",
        "-8.4375,-3.07980097018721557079482134568e-5",
        "-8.3125,-4.77308821668490236681041867788e-5",
        "-8.1875,-9.36886319777391150689116057684e-5",
        "-8.0625,-3.49293006478083171347016812056e-4",
        "-7.9375,4.56455033434365371163705444074e-4",
        "-7.8125,2.0907375836236570137155309836e-4",
        "-7.6875,1.81878551999663430045051825546e-4",
        "-7.5625,2.00362061715103622052852756283e-4",
        "-7.4375,2.59858206859546313785813051041e-4",
        "-7.3125,3.96762958011932509241116052598e-4",
        "-7.1875,7.67075674317739004626713772229e-4",
        "-7.0625,2.8161748647295455689853230472e-3",
        "-6.9375,-3.62311182788527513361191196234e-3",
        "-6.8125,-1.63338873720598204196525858093e-3",
        "-6.6875,-1.39819136849741261847133590888e-3",
        "-6.5625,-1.51523809172047114177469896939e-3",
        "-6.4375,-1.93269541351787570878198456712e-3",
        "-6.3125,-2.90132913046225647382566113463e-3",
        "-6.1875,-5.5133564091587490957545052379e-3",
        "-6.0625,-1.98892349821524155809588440209e-2",
        "-5.9375,2.51353383059540962394326392387e-2",
        "-5.8125,1.11274607722157526608883240826e-2",
        "-5.6875,9.35040477682644688602705889066e-3",
        "-5.5625,9.9437499769155918678964619866e-3",
        "-5.4375,1.24417267245213248752840256508e-2",
        "-5.3125,1.83146401360429939910244859123e-2",
        "-5.1875,3.41138927816697600299810011595e-2",
        "-5.0625,1.20578487079299019459562991877e-1",
        "-4.9375,-1.4924107119160244642163129548e-1",
        "-4.8125,-6.46783657385040623414133837302e-2",
        "-4.6875,-5.31804271682004166642788974406e-2",
        "-4.5625,-5.53121092465929797651740698005e-2",
        "-4.4375,-6.76518890645847040093568894764e-2",
        "-4.3125,-9.72965257227284055773175814092e-2",
        "-4.1875,-1.76965818804911880155526443515e-1",
        "-4.0625,-6.10428590838951286014037646375e-1",
        "-3.9375,7.36877789008537079206804521432e-1",
        "-3.8125,3.11264635116550800018051909202e-1",
        "-3.6875,2.49283252350939453113807331753e-1",
        "-3.5625,2.52361498437580470178606693465e-1",
        "-3.4375,3.00205257724094624041521197052e-1",
        "-3.3125,4.19591267179266249052182069827e-1",
        "-3.1875,7.41044366245568498151266982218e-1",
        "-3.0625,2.4798661502832395994320279384e0",
        "-2.9375,-2.90145629422111474937679280314e0",
        "-2.8125,-1.18669642138184992506882290383e0",
        "-2.6875,-9.19231993044089233357164535839e-1",
        "-2.5625,-8.99037838183880425011286345468e-1",
        "-2.4375,-1.03195557342657527014272911486e0",
        "-2.3125,-1.3898960725313194499853531063e0",
        "-2.1875,-2.36207891740774958785716350582e0",
        "-2.0625,-7.59459008524242127326058556134e0",
        "-1.9375,8.52302786427452457629432885922e0",
        "-1.8125,3.33758368513645291425606441703e0",
        "-1.6875,2.47043598130598981464737969007e0",
        "-1.5625,2.30378446034619358909142126026e0",
        "-1.4375,2.51539171022727722097290221748e0",
        "-1.3125,3.21413466772867622809112905833e0",
        "-1.1875,5.16704763182945222343754516898e0",
        "-1.0625,1.56638420508124938760999577203e1",
        "-0.9375,-1.65133664870318913665702621647e1",
        "-0.8125,-6.04937042930982090708911675586e0",
        "-0.6875,-4.16886071845385781221745322699e0",
        "-0.5625,-3.59966321929092748295534571916e0",
        "-0.4375,-3.61587558345171100514854693763e0",
        "-0.3125,-4.21855175139388754936960688905e0",
        "-0.1875,-6.13586906279747451533208488817e0",
        "-0.0625,-1.66428321789882747433562050778e1",
        "0.0625,1.54812810815923981561596207794e1",
        "0.1875,4.91511347381422948700990736413e0",
        "0.3125,2.86609174393702724589949909355e0",
        "0.4375,2.02481056085114670916238196703e0",
        "0.5625,1.58194556776012356475248928521e0",
        "0.6875,1.31829742231058985917800215283e0",
        "0.8125,1.15047544927452647162476591653e0",
        "0.9375,1.04017701118676717145976281736e0",
        "1.0625,9.67580067599524884759976298715e-1",
        "1.1875,9.21583776340168028814357630775e-1",
        "1.3125,8.95653669980321014343593466736e-1",
        "1.4375,8.85854620372376685258542110574e-1",
        "1.5625,8.89844381865069505173275222933e-1",
        "1.6875,9.0632947783853052818487648007e-1",
        "1.8125,9.34761302535552758195122307182e-1",
        "1.9375,9.75165947987594223243527641277e-1",
        "2.0625,1.02805382182449519005747481739e0",
        "2.1875,1.09438073440394953421704968655e0",
        "2.3125,1.17554544184917133132596642509e0",
        "2.4375,1.27341601678529148505915428395e0",
        "2.5625,1.39038184666417110183324253583e0",
        "2.6875,1.52943099385252026631197906012e0",
        "2.8125,1.69425486084568937422865918177e0",
        "2.9375,1.88938402422596380753433480497e0",
        "3.0625,2.12036100751302132949354181086e0",
        "3.1875,2.39395785650863960609979618932e0",
        "3.3125,2.71844883427620870369129735802e0",
        "3.4375,3.10395154091414799483168856713e0",
        "3.5625,3.56285348207693844844768399807e0",
        "3.6875,4.11034579597864821571344372407e0",
        "3.8125,4.76509179612850136501810394872e0",
        "3.9375,5.55006557116376868463210848961e0",
        "4.0625,6.49360558550862782157397179575e0",
        "4.1875,7.63074066762128874444310035345e0",
        "4.3125,9.00486176353994133097742249845e0",
        "4.4375,1.06698334218923837322339294495e1",
        "4.5625,1.26926655298990932225948742431e1",
        "4.6875,1.51569001226712652954433237325e1",
        "4.8125,1.81669124727399114541315213045e1",
        "4.9375,2.18533831864573391957389271778e1",
        "5.0625,2.63802726911288005251442604202e1",
        "5.1875,3.19537265456641466173554827301e1",
        "5.3125,3.88334663552659969898401345246e1",
        "5.4375,4.73473858096474528117880619322e1",
        "5.5625,5.79102864801646128280891137343e1",
        "5.6875,7.10479693250215560723905799961e1",
        "5.8125,8.74282662750608238730079462779e1",
        "5.9375,1.07901079483133112278960952941e2",
        "6.0625,1.33550130498839552658542818377e2",
        "6.1875,1.65759956455632760577531566662e2",
        "6.3125,2.06302790012350609008525714662e2",
        "6.4375,2.57451410339958024664097586756e2",
        "6.5625,3.22125968545915658856245695147e2",
        "6.6875,4.04085325536060100161721423728e2",
        "6.8125,5.0817679772379103876185868774e2",
        "6.9375,6.40662659431102854156330658085e2",
        "7.0625,8.09647666149214787992415836413e2",
        "7.1875,1.02563973056922770607347656872e3",
        "7.3125,1.3022863619529632193663185738e3",
        "7.4375,1.65734345406347978377512821474e3",
        "7.5625,2.1139516685825715112441123744e3",
        "7.6875,2.70232061452240191983151202118e3",
        "7.8125,3.46195443449332645156516231023e3",
        "7.9375,4.44459719980327605070954394046e3",
        "8.0625,5.71813664217882944019643684467e3",
        "8.1875,7.3717855634663241374031128377e3",
        "8.3125,9.52296902178104354161620457093e3",
        "8.4375,1.23264919395971308918275160972e4",
        "8.5625,1.59867594936556970537835998314e4",
        "8.6875,2.07740897241409647587047486628e4",
        "8.8125,2.70465190194791129028528305487e4",
        "8.9375,3.52789902734385036525070050274e4",
        "9.0625,4.61024766775668123615837720601e4",
        "9.1875,6.03564943008805288749879863586e4",
        "9.3125,7.91596799935549244396847004958e4",
        "9.4375,1.0400477574035079189979466707e5",
        "9.5625,1.36886628164426906023022073556e5",
        "9.6875,1.80474904478474631341247504008e5",
        "9.8125,2.3834744885915968245639056921e5",
        "9.9375,3.15305975568856626394281357433e5",
        "10.0625,4.17803694890449237026852934295e5",
        "10.1875,5.5452529138933985903895212467e5",
        "10.3125,7.37174519939980233844563773368e5",
        "10.4375,9.81545071049560598554312170471e5",
        "10.5625,1.30897838182233228884514857838e6",
        "10.6875,1.74835063713522299111833519508e6",
        "10.8125,2.33878434193050438410333246037e6",
        "10.9375,3.13335313221551272479317098949e6",
        "11.0625,4.20414967983514544758270765134e6",
        "11.1875,5.64922640602889981395932477008e6",
        "11.3125,7.60211223688104616152206391285e6",
        "11.4375,1.02448766790797887474106332793e7",
        "11.5625,1.38260841579983848009268818592e7",
        "11.6875,1.86854974343826957175772073974e7",
        "11.8125,2.52881056971235786531172822278e7",
        "11.9375,3.42710498836071704274253076975e7",
        "12.0625,4.6508405833176296513883703393e7",
        "12.1875,6.32007204174483166686699458652e7",
        "12.3125,8.59988946797168347022183480141e7",
        "12.4375,1.17175777016975083798509118132e8",
        "12.5625,1.59864098076856324260717071497e8",
        "12.6875,2.18386751264347756199183611457e8",
        "12.8125,2.98715748547272272839947896316e8",
        "12.9375,4.09110657985560596977389610639e8",
        "13.0625,5.61007645362689076698722172178e8",
        "13.1875,7.70258780087651359399414965232e8",
        "13.3125,1.05886139074401352727106340992e9",
        "13.4375,1.45737372664862760474395715677e9",
        "13.5625,2.00829273209050757352525821068e9",
        "13.6875,2.77078190666641215677714207036e9",
        "13.8125,3.82729552826192599576183242155e9",
        "13.9375,5.29286913768819022339497808764e9",
        "14.0625,7.32816236755012606437705837407e9",
        "14.1875,1.0157787662405902302079784854e10",
        "14.3125,1.40960922642796800817960316446e10",
        "14.4375,1.9583459451840933438746924294e10",
        "14.5625,2.72374701789775089659363144823e10",
        "14.6875,3.79250773474965163958871320881e10",
        "14.8125,5.28645194841178528164603103226e10",
        "14.9375,7.37693636065291512385675070965e10",
        "15.0625,1.03052283293673647780302383385e11",
        "15.1875,1.44113612460383738910756947616e11",
        "15.3125,2.01750320532502921170705702914e11",
        "15.4375,2.82736195835953476521908719495e11",
        "15.5625,3.96645659481359974316447579649e11",
        "15.6875,5.57024573541355084564592252544e11",
        "15.8125,7.83055694858495694843818346654e11",
        "15.9375,1.10192986887252919662610213725e12",
        "16.0625,1.55222501711095931969080464974e12",
        "16.1875,2.18872548924207803470712114192e12",
        "16.3125,3.08930178315395098042643107586e12",
        "16.4375,4.36474002321753179380696585721e12",
        "16.5625,6.17279807567866460029971545828e12",
        "16.6875,8.73832299743000788910704096179e12",
        "16.8125,1.23820681749499631747178776065e13",
        "16.9375,1.75620072851559340712285028125e13",
        "17.0625,2.49326143373447840725335496865e13",
        "17.1875,3.54299938571061381868215234848e13",
        "17.3125,5.0394235337698825368206156925e13",
        "17.4375,7.17454141316381788607020012778e13",
        "17.5625,1.02236968128427882442464037278e14",
        "17.6875,1.4582076501961325664947374605e14",
        "17.8125,2.08173521191346255874944317259e14",
        "17.9375,2.97456498392328633331432766387e14",
        "18.0625,4.25412732130945378237603691526e14",
        "18.1875,6.08953019419011750085994934896e14",
        "18.3125,8.72450199283910914187069091765e14",
        "18.4375,1.25106065892044074388349114728e15",
        "18.5625,1.79553675275551468539577465469e15",
        "18.6875,2.57920478128440947698756688326e15",
        "18.8125,3.70809084622085518277244565117e15",
        "18.9375,5.33562593991239486038257524706e15",
        "19.0625,7.68401747411520089441671667818e15",
        "19.1875,1.10753330406832762046890328784e16",
        "19.3125,1.59767442743866186160507027429e16",
        "19.4375,2.3066430898845626215351868028e16",
        "19.5625,3.33296509730242413476590670277e16",
        "19.6875,4.81988893502524021012051561308e16",
        "19.8125,6.97584590445298381259066338127e16",
        "19.9375,1.01043416237090977668495018741e17",
        "20.0625,1.46476583100321017049818661678e17",
        "20.625,7.86500896128685297200838843101e17",
        "21.125,3.55029442322092508739165194691e18",
        "21.625,1.6221580982654134254767301139e19",
        "22.125,7.49999696905420424711486473785e19",
        "22.625,3.5079168874989565325934288713e20",
        "23.125,1.65937432940324268967416382325e21",
        "23.625,7.93666195796638915499263282131e21",
        "24.125,3.83730313674499871987150384127e22",
        "24.625,1.87503638756955943786700950404e23",
        "25.125,9.25749381739730941169000301705e23",
        "25.625,4.61727710439004011574751090369e24",
        "26.125,2.32594532162107398968711325803e25",
        "26.625,1.18317725799994777966029966907e26",
        "27.125,6.07653215273505579805758338661e26",
        "27.625,3.1502094494248609633455478689e27",
        "28.125,1.64825934642938388522311949362e28",
        "28.625,8.70245360403617841124207598783e28",
        "29.125,4.6357294118326421771900235758e29",
        "29.625,2.49107734415535607021804425152e30",
        "30.125,1.35015619119625703410659436645e31",
        "30.625,7.37981663206024235802095609512e31",
        "31.125,4.06734552597872431524611552894e32",
        "31.625,2.26006884356844922214391780413e33",
        "32.125,1.26596129496087794312035345838e34",
        "32.625,7.14746771778522066503014005556e34",
        "33.125,4.06690066006182039227413548505e35",
        "33.625,2.33186134292742824196608319313e36",
        "34.125,1.34716084364547800494080737942e37",
        "34.625,7.84088376559347746361095473689e37",
        "35.125,4.59718637894019369186050518228e38",
        "35.625,2.71490600383674157177529307765e39",
        "36.125,1.61476171560274303426600244528e40",
        "36.625,9.67185263866839184944948158912e40",
        "37.125,5.83332669761490921128593383356e41",
        "37.625,3.54231602891229851486087263202e42",
        "38.125,2.16562253648953504468990293571e43",
        "38.625,1.3327964058782523162164033278e44",
        "39.125,8.2564359203663523578802549424e44",
        "39.625,5.14792611770474957138585785361e45",
        "40.125,3.23033055384333536002064974621e46",
        "40.625,2.03986572414050701766164617449e47",
        "41.125,1.29617013472963831320828571067e48",
        "41.625,8.28695450432080975925043758388e48",
        "42.125,5.33049967907563756306907498512e49",
        "42.625,3.44944481242353706228799464429e50",
        "43.125,2.24547298981061232344284783748e51",
        "43.625,1.47032585129553267280025771713e52",
        "44.125,9.68360226855826564484728129914e52",
        "44.625,6.41429652627676128509112429097e53",
        "45.125,4.27288950100133471578886287325e54",
        "45.625,2.86237982485100472347191421485e55",
        "46.125,1.92814138732685229049972437155e56",
        "46.625,1.30596079508827090508406086052e57",
        "47.125,8.89355214904510618992997866379e57",
        "47.625,6.08904220709906309495443376219e58",
        "48.125,4.19108645023750629200450244531e59",
        "48.625,2.89990635113092879897204907924e60",
        "49.125,2.01696035417679990302716680181e61",
        "49.625,1.41007946323741412850015886478e62",
        "50.125,9.90831773989352952362095691387e62",
        "50.625,6.99751933631566761268203836648e63",
        "51.125,4.96654426712163167371500465308e64",
        "51.625,3.54249416400980672892028192303e65",
        "52.125,2.53914575656593419318679612889e66",
        "52.625,1.82881261217006272380509554277e67",
        "53.125,1.32352972560999319819861748218e68",
        "53.625,9.6241263715449550840243152938e68",
        "54.125,7.03125166730308886543015537409e69",
        "54.625,5.1609377667409821638080390763e70",
        "55.125,3.80566496492779684841407159623e71",
        "55.625,2.81916225508226150698014134543e72",
        "56.125,2.09787281191644801268825696742e73",
        "56.625,1.5681590043895079632577036234e74",
        "57.125,1.17743111568810644712128422296e75",
        "57.625,8.87970036235558884194674676748e75",
        "58.125,6.72607524836830807918033612368e76",
        "58.625,5.11692733380740807017181282476e77",
        "59.125,3.90953123811407907102357037189e78",
        "59.625,2.99979864944459298113822526851e79",
        "60.125,2.31151034453494925074268598238e80",
        "60.625,1.78862994473133856500366681635e81",
        "61.125,1.38979559465163823700903994691e82",
        "61.625,1.08435690399337400503347300741e83",
        "62.125,8.49512557230813872371775667546e83",
        "62.625,6.68234942085916730601877740818e84",
        "63.125,5.27759676179643118210965633463e85",
        "63.625,4.18482132481305352539425935187e86",
        "64.125,3.33148295588399718370672056124e87",
        "64.625,2.66259256791230530553209751263e88",
        "65.125,2.13631344546061319405193455989e89",
        "65.625,1.72070044701332730370011801754e90",
        "66.125,1.39127413135622434262632238213e91",
        "66.625,1.12920966835249604305320244901e92",
        "67.125,9.19980019359303346561655675184e92",
        "67.625,7.52335941539850488684196131652e93",
        "68.125,6.17536587994932371379511371967e94",
        "68.625,5.0876718046632389297268763403e95",
        "69.125,4.20696800571547678002292122153e96",
        "69.625,3.49141477595014771552506888853e97",
        "70.125,2.90806663395082332419084429438e98",
        "70.625,2.43089753775529034693432921364e99",
        "71.125,2.03928172705801485608882956143e100",
        "71.625,1.71682138603967380752237000713e101",
        "72.125,1.45043912837001306639318002557e102",
        "72.625,1.22967331775091636463789751761e103",
        "73.125,1.04612922133687192413608109344e104",
        "73.625,8.93050247016603009818273072163e104",
        "74.125,7.6498199310258759452450929958e105",
        "74.625,6.5750824436597396597870354938e106",
        "75.125,5.67042902387293054441292518313e107",
        "75.625,4.90665527358108072111607523725e108",
        "76.125,4.25990980418453907149021004383e109",
        "76.625,3.71065805064569229534403189817e110",
        "77.125,3.24285633843548036817192239586e111",
        "77.625,2.84329173130726172130736444197e112",
        "78.125,2.50105295101836423395259514781e113",
        "78.625,2.20710520642726191116484164808e114",
        "79.125,1.95394761798309705777546495923e115",
        "79.625,1.7353364685534346776533567458e116",
        "80.125,1.54606105272912554696483664899e117",
        "80.625,1.38176166308567236208148530885e118",
        "81.125,1.238781418499211844505575365e119",
        "81.625,1.11404534086282334192819753026e120",
        "82.125,1.00496142575748560885514801486e121",
        "82.625,9.09339509479279552848891234073e121",
        "83.125,8.25324570903335056272290307202e122",
        "83.625,7.51341769707254730541396382152e123",
        "84.125,6.86051049563397265526341317862e124",
        "84.625,6.28309554917691768415242724575e125",
        "85.125,5.77140445445207949624034633651e126",
        "85.625,5.31706960849096659021399155672e127",
        "86.125,4.91290804185233267117459481896e128",
        "86.625,4.55274085227039014287073027044e129",
        "87.125,4.23124205104532151304911978783e130",
        "87.625,3.94381176327922546126177009677e131",
        "88.125,3.68646963697323636824404561514e132",
        "88.625,3.45576505757342131043062604729e133",
        "89.125,3.24870136758266454951506519834e134",
        "89.625,3.06267178227444463636914233441e135",
        "90.125,2.89540509385804977975530185802e136",
        "90.625,2.74491958486347100534584381722e137",
        "91.125,2.60948384083956736400446579954e138",
        "91.625,2.48758337378252059859467095935e139",
        "92.125,2.37789214996505576044906945984e140",
        "92.625,2.27924826622823449846236726651e141",
        "93.125,2.19063314315530761931370523987e142",
        "93.625,2.1111537065939022042007676806e143",
        "94.125,2.04002711456338022048588800463e144",
        "94.625,1.97656765779854093868296874096e145",
        "95.125,1.92017552158278163253234208436e146",
        "95.625,1.87032714619186936322875917114e147",
        "96.125,1.82656696490562102794639040775e148",
        "96.625,1.7885003335459750785875009574e149",
        "97.125,1.75578749501552821311346777945e150",
        "97.625,1.72813844728879841968517280009e151",
        "98.125,1.70530860453383177698645558079e152",
        "98.625,1.68709515916568945721764994609e153",
        "99.125,1.67333406819882243116795953865e154",
        "99.625,1.66389760072716122718090725933e155",
        "100.125,1.65869239510208273489523989268e156",
    })
    void testGamma(double x, double expected) {
        final double actual = Gamma.value(x);
        int ulps;
        if (Math.abs(x) <= 8.0) {
            ulps = 2;
        } else {
            ulps = 4;
        }
        final double tol = ulps * Math.ulp(expected);
        Assertions.assertEquals(expected, actual, tol, () -> Double.toString(x));
    }

    @Test
    void testGammaNegativeInteger() {
        for (int i = -100; i <= 0; i++) {
            final int x = i;
            Assertions.assertTrue(Double.isNaN(Gamma.value(x)), () -> Integer.toString(x));
        }
    }

    @Test
    void testGammaNegativeDouble() {
        // check that the gamma function properly switches sign
        // see: https://en.wikipedia.org/wiki/Gamma_function

        double previousGamma = Gamma.value(-18.5);
        for (double x = -19.5; x > -25; x -= 1.0) {
            final double gamma = Gamma.value(x);
            Assertions.assertEquals((int) Math.signum(previousGamma), -(int) Math.signum(gamma));
            previousGamma = gamma;
        }
    }
}
