/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Original Copyright (c) 2005 Covalent Technologies
 *
 * FTP Protocol module for Apache 2.0
 */

#include "mod_ftp.h"
#include "ftp_internal.h"
#include "apr_portable.h"

#if APR_HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_NETINET_IP_H
#include <netinet/ip.h>
#endif

/*
 * ftp_reset_dataconn: Close any data channel listen/connect socket and
 *                     clear all data channel state from ftp_connection
 */
void ftp_reset_dataconn(ftp_connection *fc)
{
    if (fc->csock) {
        apr_socket_close(fc->csock);
        fc->csock = NULL;
    }
    fc->clientsa = NULL;
    fc->passive_created = -1;
    apr_pool_clear(fc->data_pool);
}

/*
 * ftp_open_datasock: If we are in passive mode we accept and return a
 *                    socket.  If we are in active mode, a socket is
 *                    created based on the fc->clientsa sockaddr and
 *                    then returned.
 *
 * Arguments: r - The request.
 *
 * Returns: apr_status_t
 */
static apr_status_t ftp_open_datasock(request_rec *r)
{
    ftp_server_config *fsc = ftp_get_module_config(r->server->module_config);
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);

    apr_interval_time_t timeout;
    apr_pollfd_t pollset[2];
    apr_socket_t *s;
    apr_status_t rv, res;
    int n;
#ifdef HAVE_SOL_IP_H
    int sd, sopt;
#endif

    /*
     * handle err condition when the creation of the socket had failed, this
     * will occur if a PORT command has failed
     */
    if (!fc->csock) {
        ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, r->server,
                     "Nonexistant connection");
        return APR_EGENERAL;
    }

    if (fc->passive_created != -1) {
        pollset[0].desc_type = APR_POLL_SOCKET;
        pollset[0].desc.s = fc->csock;
        pollset[0].reqevents = APR_POLLIN;
        pollset[1].desc_type = APR_POLL_SOCKET;
        pollset[1].desc.s = fc->cntlsock;
        pollset[1].reqevents = (APR_POLLIN | APR_POLLPRI);

        /*
         * since it is possible to hang in accept(), poll both the control
         * and client sockets waiting for activity.
         */
        apr_socket_timeout_get(fc->csock, &timeout);    /* likely to be -1 */
        res = apr_poll(pollset, 2, &n, timeout);

        if (res == APR_SUCCESS) {
            if (pollset[0].rtnevents & APR_POLLIN) {
                /* activity on client socket, fall through to accept() */
                ;
            }
            else if (pollset[1].rtnevents & (APR_POLLIN | APR_POLLPRI)) {
                /*
                 * command channel has activity. since we can only do a
                 * single read ahead operation, no use in looping here if the
                 * command is not an ABOR.  bail out and let command
                 * processing occur as normal.
                 */
                ap_log_error(APLOG_MARK, APLOG_ERR, res, r->server,
                             "Activity on control channel while waiting "
                             "for client connect, processing command");

                /* manual cleanup, no need to close the socket */
                fc->csock = NULL;
                fc->passive_created = -1;
                return APR_ECONNRESET;
            }
        }
        else {
            /*
             * not much we can do, one of our sockets was likely disconnected
             */
            fc->csock = NULL;
            fc->passive_created = -1;
            return APR_EGENERAL;
        }

        /* activity on client socket, attempt an accept() */
        rv = apr_socket_accept(&s, fc->csock, fc->data_pool);

        res = apr_socket_close(fc->csock);
        fc->csock = NULL;
        fc->passive_created = -1;

        /* check csock closure first */
        if (res != APR_SUCCESS) {
            ap_log_error(APLOG_MARK, APLOG_ERR, res, r->server,
                         "Couldn't close passive connection");
        }

        /* retest the apr_accept result from above */
        if (rv != APR_SUCCESS) {
            return rv;
        }
        fc->datasock = s;
    }
    else {
        if (fc->clientsa) {
            int tries;
            for (tries = 0;; tries++) {
                rv = apr_socket_connect(fc->csock, fc->clientsa);
                if (rv == APR_SUCCESS) {
                    break;
                }
                if (!APR_STATUS_IS_EAGAIN(rv) || tries > FTP_MAX_TRIES) {
                    ap_log_error(APLOG_MARK, APLOG_ERR, rv, r->server,
                                 "Couldn't connect to client");
                    apr_socket_close(fc->csock);
                    fc->csock = NULL;
                    fc->passive_created = -1;
                    return rv;
                }
                apr_sleep(tries * APR_USEC_PER_SEC);
            }
        }
        else {
            return APR_EGENERAL;
        }
        fc->datasock = fc->csock;
        fc->csock = NULL;
        fc->passive_created = -1;
    }

#ifdef HAVE_SOL_IP_H
    sopt = IPTOS_THROUGHPUT;
    if (((apr_os_sock_get(&sd, fc->datasock)) == APR_SUCCESS) &&
#ifdef HAVE_SOL_IP
         (setsockopt(sd, SOL_IP, IP_TOS, &sopt, sizeof(sopt)) < 0)) {
#else
         (setsockopt(sd, IPPROTO_IP, IP_TOS, &sopt, sizeof(sopt)) < 0)) {
#endif
        ap_log_error(APLOG_MARK, APLOG_ERR, errno, r->server,
                     "Failed to set TOS priority");
    }
#endif

    rv = apr_socket_opt_set(fc->datasock, APR_SO_LINGER,
                            APR_MAX_SECS_TO_LINGER);
    if (rv != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, r->server,
                     "Couldn't set APR_SO_LINGER socket option");
    }
    rv = apr_socket_opt_set(fc->datasock, APR_SO_REUSEADDR, 1);
    if (rv != APR_SUCCESS && rv != APR_ENOTIMPL) {
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, r->server,
                     "Couldn't set APR_SO_REUSEADDR socket option");
    }

    /* Set the default data connection timeout value */
    rv = apr_socket_timeout_set(fc->datasock,
                                fsc->timeout_data * APR_USEC_PER_SEC);
    if (rv != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, r->server,
                     "Couldn't set socket timeout");
    }

    return APR_SUCCESS;
}

/*
 * ftp_open_dataconn: Creates the appropriate data channel
 *                    and initializes the
 *
 * Arguments: r - The request.
 *            write - Open for write-to-client only
 *
 * Returns: apr_status_t
 */
conn_rec *ftp_open_dataconn(request_rec *r, int write_not_read)
{
    ftp_server_config *fsc = ftp_get_module_config(r->server->module_config);
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    conn_rec *cdata;
    ap_filter_t *f;

    if (ftp_open_datasock(r) != APR_SUCCESS) {
        return NULL;
    }

    cdata = ap_run_create_connection(r->pool, r->server, fc->datasock,
                                     r->connection->id, r->connection->sbh,
                                     r->connection->bucket_alloc);

    ftp_set_module_config(cdata->conn_config, fc);

    ap_run_pre_connection(cdata, fc->datasock);

    /*
     * Open data connection only if the connection is from the client's IP
     * address. All other PASV connection attempts are denied, unless
     * disabled using:
     * 
     * FTPOptions AllowProxyPASV
     */
    if (fc->clientsa == NULL) { /* Only check PASV, never PORT connections */
        if (!(fsc->options & FTP_OPT_ALLOWPROXYPASV)) {
            if (strcmp(fc->connection->remote_ip, cdata->remote_ip) != 0) {
                ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, r->server,
                             "PASV data connection attempt from %s "
                             "doesn't match the client IP %s",
                             cdata->remote_ip, fc->connection->remote_ip);
                ap_log_error(APLOG_MARK, APLOG_ERR, 0, r->server,
                             "PASV data connection attempt denied, "
                             "not configured to AllowProxyPASV");
                apr_socket_close(fc->datasock);
                fc->datasock = NULL;
                return NULL;
            }
        }
        fc->passive_created = 0;
    }

    if (write_not_read) {
        /*
         * We need the network-level poll filter to watch both the control
         * incoming command and writable data port conditions
         */
        fc->filter_mask += FTP_NEED_DATA_OUT;
    }
    else {
        /*
         * Once Apache has inserted the core i/o filters, we must insert our
         * idea of a core socket, based on our own ftp_datasock bucket,
         * instead of the socket_bucket. This will capture any abort command
         * on the control socket while actually reading from the data socket.
         * 
         * Insert this bucket type only for read connections
         */
        for (f = cdata->input_filters; f; f = f->next) {
            if (strcasecmp(f->frec->name, "CORE_IN") == 0) {
                core_net_rec *net = f->ctx;
                apr_bucket *e;

                net->in_ctx = apr_pcalloc(fc->data_pool, sizeof(*net->in_ctx));
                net->in_ctx->b = apr_brigade_create(fc->data_pool,
                                                    f->c->bucket_alloc);
                net->in_ctx->tmpbb =
                    apr_brigade_create(net->in_ctx->b->p,
                                       net->in_ctx->b->bucket_alloc);

                /* seed the brigade with our client data+control sockets */
                e = ftp_bucket_datasock_create(fc, f->c->bucket_alloc);
                APR_BRIGADE_INSERT_TAIL(net->in_ctx->b, e);
                break;
            }
        }
    }

    /*
     * We initalize the data connection here by adding/removing the SSL/TLS
     * filters.  We then invoke ftp_ssl_init immediately for writers, since
     * negotation would normally occur on the first socket read (and we won't
     * be reading from that socket.)
     */
    if (fc->prot == FTP_PROT_CLEAR) {
        for (f = cdata->output_filters; f; f = f->next) {
            if (strcasecmp(f->frec->name, FTP_SSL_FILTER) == 0) {
                ap_remove_output_filter(f);
            }
        }
        for (f = cdata->input_filters; f; f = f->next) {
            if (strcasecmp(f->frec->name, FTP_SSL_FILTER) == 0) {
                ap_remove_input_filter(f);
            }
        }
    }
    else if ((fc->prot == FTP_PROT_PRIVATE) && write_not_read) {
        if (ftp_ssl_init(cdata) != APR_SUCCESS) {
            /*
             * In case of failure within ssl_init, return no data connection
             */
            apr_socket_close(fc->datasock);
            fc->datasock = NULL;
            return NULL;
        }
    }

    /*
     * We now need to remove the NET_TIME filter to allow
     * use to control timeouts ourselves.
     */
    for (f = cdata->input_filters; f; f = f->next) {
        if (strcasecmp(f->frec->name, "NET_TIME") == 0) {
            ap_remove_input_filter(f);
        }
    }

    return cdata;
}
