"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getFormInputsAsArr = exports.fixEmptyParameters = exports.resolveFiles = exports.decodeEncodedValues = exports.encodeFormData = void 0;
const mona_dish_1 = require("mona-dish");
const ExtDomQuery_1 = require("./ExtDomQuery");
const Const_1 = require("../core/Const");
/*
 * various routines for encoding and decoding url parameters
 * into configs and vice versa
 */
/**
 * encodes a given form data into a url encoded string
 * @param formData the form data config object
 * @param paramsMapper the params mapper
 * @param defaultStr a default string if nothing comes out of it
 */
function encodeFormData(formData, paramsMapper = (inStr, inVal) => [inStr, inVal], defaultStr = Const_1.EMPTY_STR) {
    if (formData.isAbsent()) {
        return defaultStr;
    }
    const assocValues = formData.value;
    const expandValueArrAndRename = key => assocValues[key].map(val => paramsMapper(key, val));
    const isPropertyKey = key => assocValues.hasOwnProperty(key);
    const isNotFile = ([, value]) => !(value instanceof ExtDomQuery_1.ExtDomQuery.global().File);
    const mapIntoUrlParam = keyVal => `${encodeURIComponent(keyVal[0])}=${encodeURIComponent(keyVal[1])}`;
    return new mona_dish_1.Es2019Array(...Object.keys(assocValues))
        .filter(isPropertyKey)
        .flatMap(expandValueArrAndRename)
        .filter(isNotFile)
        .map(mapIntoUrlParam)
        .join("&");
}
exports.encodeFormData = encodeFormData;
/**
 * splits and decodes encoded values into strings containing of key=value
 * @param encoded encoded string
 */
function decodeEncodedValues(encoded) {
    const filterBlanks = item => !!(item || '').replace(/\s+/g, '');
    const splitKeyValuePair = _line => {
        let line = decodeURIComponent(_line);
        let index = line.indexOf("=");
        if (index == -1) {
            return [line];
        }
        return [line.substring(0, index), line.substring(index + 1)];
    };
    let requestParamEntries = encoded.split(/&/gi);
    return requestParamEntries.filter(filterBlanks).map(splitKeyValuePair);
}
exports.decodeEncodedValues = decodeEncodedValues;
/**
 * gets all the input files and their corresponding file objects
 * @param dataSource
 */
function resolveFiles(dataSource) {
    const expandFilesArr = ([key, files]) => {
        return [...files].map(file => [key, file]);
    };
    const remapFileInput = fileInput => {
        return [fileInput.name.value || fileInput.id.value, fileInput.filesFromElem(0)];
    };
    const files = dataSource
        .querySelectorAllDeep("input[type='file']")
        .asArray;
    const ret = files
        .map(remapFileInput)
        .flatMap(expandFilesArr);
    return ret;
}
exports.resolveFiles = resolveFiles;
function fixEmptyParameters(keyVal) {
    var _a, _b;
    return (keyVal.length < 3 ? [(_a = keyVal === null || keyVal === void 0 ? void 0 : keyVal[0]) !== null && _a !== void 0 ? _a : [], (_b = keyVal === null || keyVal === void 0 ? void 0 : keyVal[1]) !== null && _b !== void 0 ? _b : []] : keyVal);
}
exports.fixEmptyParameters = fixEmptyParameters;
/**
 * returns the decoded viewState from parentItem
 * @param parentItem
 */
function resolveViewState(parentItem) {
    const viewStateStr = (0, Const_1.$faces)().getViewState(parentItem.getAsElem(0).value);
    // we now need to decode it and then merge it into the target buf
    // which hosts already our overrides (aka do not override what is already there(
    // after that we need to deal with form elements on a separate level
    return decodeEncodedValues(viewStateStr);
}
/**
 * gets all the inputs under the form parentItem
 * as array
 * @param parentItem
 */
function getFormInputsAsArr(parentItem) {
    const standardInputs = resolveViewState(parentItem);
    const fileInputs = resolveFiles(parentItem);
    return standardInputs.concat(fileInputs);
}
exports.getFormInputsAsArr = getFormInputsAsArr;
//# sourceMappingURL=FileUtils.js.map